import java.util.Arrays;

class Rasterizer {
    // private Referenz auf die Szene
    private final Scene scene;

    // private Listen der zu zeichnenden Linien und ihrer
    // Endpunkte, der Vertices
    private final ArrayList<Point3D> vertices;
    private final ArrayList<Line> lines;

    // der Konstruktor speichert eine private Referenz auf die Szene
    // und legt die Linien- und Endpunktlisten an
    Rasterizer(Scene scene) {
        this.scene = scene;

        vertices = new ArrayList<Point3D>();
        lines = new ArrayList<Line>();
    }

    // füge die Linien aller Wireframe-Modelle in der Szene
    // dem Rasterizer hinzu
    void stageLines() {
        // in jedem Frame der Animation können
        // unterschiedliche Punkte und Linien sichtbar sein;
        // daher müssen wir die Listen zu Beginn immer löschen
        vertices.clear();
        lines.clear();

        // wir bearbeiten alle Wireframe-Modelle in der Szene
        for (WireFrame wf : scene.wireFrames) {

            // in den Wireframe-Modellen starten die Punktnummern
            // immer bei 0; hier werden sie jedoch hintereinander
            // eingefügt; wir müssen die bisherige Anzahl an
            // Punkten also speichern
            int numOfVertices = vertices.size();

            // die maximale z-Koordinate eines WireFrames
            float maxZ = Float.NEGATIVE_INFINITY;

            // jetzt wird die Model-Transformation auf alle Punkte
            // angewandt und das Ergebnis eingefügt
            for (Point3D v : wf.vertices) {
                Point3D p = wf.applyModelTransform(v);
                vertices.add(p);
                
                // falls die z-Koordinate größer als die bisher maximale
                // ist, speichern wir sie
                maxZ = max(maxZ, p.z);
            }

            // zum Schluss werden die Linien eingefügt und die Punktnummern angepasst

            // wenn die maximale z-Koordinate des Wireframe-Modells vor der Kamera liegt,
            // können wir die Linien einfach so einfügen
            if (maxZ <= -1.0) {
                for (Line l : wf.lines) {
                    lines.add(new Line(l.v0 + numOfVertices, l.v1 + numOfVertices, l.pigment));    
                }
            }
            // wenn eine z-Koordinate hinter dem Schirm liegt, müssen wir
            // die Linie ev. abschneiden
            else {
                for (Line l : wf.lines) {
                    clipLine(new Line(l.v0 + numOfVertices, l.v1 + numOfVertices, l.pigment));
                }
            }
        }
    }
    
    // überprüft, ob eine Linie abgeschnitten werden muss, weil sie nicht
    // komplett vor dem Schirm liegt; falls ja, wird die Linie entsprechend
    // angepasst
    private void clipLine(Line l) {

        // lade die Endpunkte der Linie l
        final Point3D v0 = vertices.get(l.v0);
        final Point3D v1 = vertices.get(l.v1);

        // v0 ist vor dem Schirm
        if (v0.z <= -1.0) {

            // v0 und v1 sind beide vor dem Schirm, wir können die Linie
            // wie sie ist hinzufügen
            if (v1.z <= -1.0) {
                lines.add(l);
            }
            // v0 ist vor, v1 ist hinter dem Schirm
            else {

                // der Schnittpunkt mit der Schirmebene wird als
                // letzter Vertex hinzugefügt
                vertices.add(getIntersectionPointWithScreen(v0, v1));

                // statt v1 muss der neue Punkt als Endpunkt verwendet werden
                lines.add(new Line(l.v0, vertices.size() - 1, l.pigment));
            }
        }
        // v0 ist hinter dem Schirm
        else {

            // v0 ist hinter, v1 ist vor dem Schirm
            if (v1.z <= -1.0) {

                // der Schnittpunkt mit der Schirmebene wird als
                // letzter Vertex hinzugefügt
                vertices.add(getIntersectionPointWithScreen(v0, v1));

                // statt v0 muss der neue Punkt als Endpunkt verwendet werden
                lines.add(new Line(vertices.size() - 1, l.v1, l.pigment));
            }
            // v0 und v1 sind beide hinter dem Schirm, wir
            // ignorieren die Linie daher
            else {
            }
        }
    }

    // A und B sind zwei Punkte, von denen einer vor, einer hinter dem Schirm
    // ist; die z-Koordinate des Schnittpunkts mit der Schirmebene muss -1
    // sein:
    // A_z + t*(B_z - A_z) == -1
    // wenn wir daraus t ausrechnen, können wir den Schnittpunkt der Linie AB
    // mit der Schirmebene zurückgeben
    private Point3D getIntersectionPointWithScreen(Point3D A, Point3D B) {
        final float t = -(1.0 + A.z) / (B.z - A.z);

        return new Point3D(A.x + t*(B.x - A.x), A.y + t*(B.y - A.y), -1.0);
    }

    // male die Szene
    void paintScene() {

        // male alle zu rasternden Linien
        for (Line l : lines) {

            // lade die Endpunkte der aktuellen Linie
            final Point3D v0 = vertices.get(l.v0);
            final Point3D v1 = vertices.get(l.v1);

            // und berechne ihre screen coordinates
            final int x0 = (int)((1.0 - v0.x/v0.z)*width/2.0);
            final int y0 = (int)((1.0 + v0.y/v0.z)*height/2.0);
            final int x1 = (int)((1.0 - v1.x/v1.z)*width/2.0);
            final int y1 = (int)((1.0 + v1.y/v1.z)*height/2.0);

            // der line-Befehl von Processing kümmert sich
            // um das Clipping links/rechts und oben/unten;
            // jede Linie wird mit der Farbe ihres Pigments gemalt
            stroke(l.pigment);
            line(x0, y0, x1, y1);
        }
    }
} //<>//
